#!/bin/bash

function LOG() {
  command /bin/echo $(date +'%FT%T') "$@"
}
function THROW() {
  code="$1"
  shift 1
  msg=("$@")
  [[ ! -z ${ERR_MSG_FILE} ]] && command /bin/echo "${msg[@]}" > ${ERR_MSG_FILE}
  exit ${code}
}
export -f LOG
export -f THROW

LOG "Execute main upgrade script"

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to persistent directory to update
  -h, --help                       display help
EOM

	exit 2
}

# New redis instance to update the redis db for the installing software
REDIS_PORT_NEW_INSTANCE=30001
function ping_new_redis() {
  timer=300
  redis_port=${1}
  is_up=false
  while true; do
      # Check if redis is up
      status=$(redis-cli -p ${redis_port} PING)
      if [[ "${status}" == 'PONG' ]]; then
          is_up=true
          break
      fi
      timer=$((timer-1))
      logger "Waiting for redis... timeout=${timer} status=${status}"
      if [[ ${timer} == 0 ]]; then
        is_up=false
        break
      fi
      sleep 1
  done
  echo ${is_up}
}

function wait_redis_persistent() {
  timer=40
  while true; do
      # Check if the write operation is finished
      status=$(redis-cli -p ${REDIS_PORT_NEW_INSTANCE} info persistence | grep -i aof_rewrite_in_progress)
      if [[ ${status} =~ ':0' ]]; then
          LOG "Redis aof_rewrite ready!"
          break
      fi
      timer=$((timer-1))
      LOG "Waiting for aof_rewrite_in_progress... timeout="${timer}
      if [[ ${timer} == 0 ]]; then
          LOG "aof_rewrite_in_progress timeout!"
          # exit 1, Not good to exit here in the main upgrade script
          break
      fi
      sleep 1
  done
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

PERSISTENT_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift

if [ ! -d "$PERSISTENT_DIRECTORY" ]
then
  LOG "The directory $PERSISTENT_DIRECTORY does not exist."
  exit 1
fi

RUNNING_RELEASE_VERSION=''
INSTALLING_RELEASE_VERSION=''
SCRIPTS_DIR=$(readlink -f $0 | xargs dirname)

RUNNING_RELEASE_VERSION_FILE=/etc/image/release.txt
INSTALLING_RELEASE_VERSION_FILE=${SCRIPTS_DIR}/../image/release.txt

if [ -f $RUNNING_RELEASE_VERSION_FILE ]
then
  RUNNING_RELEASE_VERSION=`cat $RUNNING_RELEASE_VERSION_FILE`
fi

if [ -z $RUNNING_RELEASE_VERSION ]
then
  LOG "Failed to get the running release version."
  exit 0
fi

if [ -f $INSTALLING_RELEASE_VERSION_FILE ]
then
  INSTALLING_RELEASE_VERSION=`cat $INSTALLING_RELEASE_VERSION_FILE`
fi

if [ -z $INSTALLING_RELEASE_VERSION ]
then
  LOG "Failed to get the installing release version."
  exit 0
fi

LOG "Running release version: $RUNNING_RELEASE_VERSION"
LOG "Installing release version: $INSTALLING_RELEASE_VERSION"

# DC-2729 Faulty time and date set on reboot
CONFIG_JSON="${PERSISTENT_DIRECTORY}/etc/smartmanager/config.json"
[[ -f ${CONFIG_JSON} ]] && sed -i '/\/system\/settings\/dateTime/d' ${CONFIG_JSON}

fail=0
PARTITION_EMMC_SCRIPT="${SCRIPTS_DIR}/emmc.sh"
if [ -f $PARTITION_EMMC_SCRIPT ]
then
  /bin/bash $PARTITION_EMMC_SCRIPT -r $RUNNING_RELEASE_VERSION -i $INSTALLING_RELEASE_VERSION -d $PERSISTENT_DIRECTORY  || fail=1
else
  LOG "The file $PARTITION_EMMC_SCRIPT does not exist."
  exit 0
fi

if [ $fail -ne 0 ]; then
  LOG "Failed to run the script $PARTITION_EMMC_SCRIPT"
  exit 1;
else
  LOG "Run the script $PARTITION_EMMC_SCRIPT successfully."
fi

fail=0
HANDLE_MMC_SCRIPT="${SCRIPTS_DIR}/handle-mmc-directories.sh"
if [ -f $HANDLE_MMC_SCRIPT ]
then
  /bin/bash $HANDLE_MMC_SCRIPT -r $RUNNING_RELEASE_VERSION -i $INSTALLING_RELEASE_VERSION -d $PERSISTENT_DIRECTORY  || fail=1
else
  LOG "The file $HANDLE_MMC_SCRIPT does not exist."
  exit 0
fi

if [ $fail -ne 0 ]; then
  LOG "Failed to run the script $HANDLE_MMC_SCRIPT"
  exit 1;
else
  LOG "Run the script $HANDLE_MMC_SCRIPT successfully."
fi

if [[ ! -f /usr/sbin/smartmanager/smartmanager.js ]] && [[ -f "/proc/device-tree/is_chassis" ]]; then
  # The current upgrade.sh script could be triggered by the dcp-release-6.x too
  #Start redis instance for new image.
  LOG "Starting new redis instance on port $REDIS_PORT_NEW_INSTANCE."
  mkdir -p "$PERSISTENT_DIRECTORY/etc/sw_storage/redis"
  redis-server /etc/redis/50001.conf --port ${REDIS_PORT_NEW_INSTANCE} --cluster-enabled no --dir "$PERSISTENT_DIRECTORY/etc/sw_storage/redis" &>/dev/null&
  REDIS_PID=$!
  sleep 1s
  PING_REDIS=`ping_new_redis ${REDIS_PORT_NEW_INSTANCE}`
  REDIS_VERSION=`redis-server --version`
  if [[ ${PING_REDIS} == true ]]; then
    LOG "Redis is up and running on port: ${REDIS_PORT_NEW_INSTANCE}, version: ${REDIS_VERSION}"
    sleep 1s # Waiting for some unexpected paths to be loaded
  else
    LOG "Failed to start redis server on port: ${REDIS_PORT_NEW_INSTANCE}, version: ${REDIS_VERSION}"
    kill -9 ${REDIS_PID}
    exit 1
  fi
  REDIS_RUNNING_NEW_INSTANCE=true
fi


fail=0

UPGRADE_SCRIPT="${SCRIPTS_DIR}/r${INSTALLING_RELEASE_VERSION}/upgrade.sh"

if [ -f $UPGRADE_SCRIPT ]
then
  /bin/bash $UPGRADE_SCRIPT -r $RUNNING_RELEASE_VERSION -i $INSTALLING_RELEASE_VERSION -d $PERSISTENT_DIRECTORY        || fail=1
else
  LOG "The file $UPGRADE_SCRIPT does not exist."
  exit 0
fi

if [ $fail -ne 0 ]
then
  LOG "Failed to run the $UPGRADE_SCRIPT script."
  exit 1;
else
  LOG "Run the upgrade script $UPGRADE_SCRIPT successfully."
fi

if [ "$REDIS_RUNNING_NEW_INSTANCE" = true ] ; then
  redis-cli -p ${REDIS_PORT_NEW_INSTANCE} BGREWRITEAOF
  wait_redis_persistent
  redis-cli -p ${REDIS_PORT_NEW_INSTANCE} SHUTDOWN SAVE
  LOG "Shutdown new redis instance port ${REDIS_PORT_NEW_INSTANCE}"
fi

# Make sure that the redis AOF file is in place. Otherwise, abort upgrade!
if [[ -f "/proc/device-tree/is_chassis" ]]; then
    DIR_NAME=$(echo $RANDOM | sha1sum | cut -c -10)
    TEST_MOUNT_POINT="/tmp/${DIR_NAME}"

    # mount the MMC into a random mountpoint to cross check
    mkdir -p ${TEST_MOUNT_POINT}
    mount /dev/mmcblk3p1 ${TEST_MOUNT_POINT}

    # sw_storage link destination of installing version
    MMC_LINK_DEST=$(basename $(readlink -f "${PERSISTENT_DIRECTORY}/etc/sw_storage"))

    # actual MMC DB file
    DB_FILE="${TEST_MOUNT_POINT}/sw_storage/${MMC_LINK_DEST}/redis/50001.aof"

    if [[ -f ${DB_FILE} ]]; then
        LOG "Redis AOF of ${INSTALLING_RELEASE_VERSION} is in place: ${DB_FILE}"
    else
        LOG "Redis AOF of ${INSTALLING_RELEASE_VERSION} is not found: ${DB_FILE}"
        if [[ -f /usr/sbin/smartmanager/smartmanager.js ]] ||
           [[ -z "${RESET_CONFIG}" ]] ||
           [[ "${RESET_CONFIG}" == "false" ]]; then
            # Abort upgrade
            fail=1
        fi
    fi

    umount ${TEST_MOUNT_POINT}
    rm -r ${TEST_MOUNT_POINT}
fi

# Try saving the log
TMP_MOUNT_POINT=/tmp/mmc-tmpfs/
if mountpoint -q ${TMP_MOUNT_POINT}
then
    mkdir -p /mnt/mmc/log/backup
    cp -r ${TMP_MOUNT_POINT}/log/* /mnt/mmc/log/backup/
    # workaround way to collect log because techlog from release 6.1 does not collect logs from MMC
    if [[ -f /usr/sbin/smartmanager/smartmanager.js ]]; then
        [[ -f ${TMP_MOUNT_POINT}/log/upgrade.log ]] && ln -fs ${TMP_MOUNT_POINT}/log/upgrade.log  /var/log/messages_upgrade.log
        [[ -f /mnt/mmc/log/persistent.log ]]        && ln -fs /mnt/mmc/log/persistent.log         /var/log/messages_mmc_persistent.log
        [[ -f /mnt/mmc/log/upgrade.log ]]           && ln -fs /mnt/mmc/log/upgrade.log            /var/log/messages_mmc_upgrade.log
    fi
    sync
fi

if [ $fail -ne 0 ]; then
  LOG "Upgrade aborted due to previous error."
  exit 1
fi

#Upgrade VSC firmware
if [ -d "/proc/device-tree/vsc751x_gpio" ]; then
  VSC_FWUPGRADE_SCRIPT="${SCRIPTS_DIR}/vscfwupgrade.sh"
  LOG "Executing the vscfwupgrade.sh"
  /bin/bash $VSC_FWUPGRADE_SCRIPT -d "${SCRIPTS_DIR}/../firmwares"  || fail=1

  if [ $fail -ne 0 ]
  then
    LOG "Failed to run the VSC firmware upgrade script: $VSC_FWUPGRADE_SCRIPT."
    exit 1;
  else
    LOG "Run the VSC firmware upgrade script $VSC_FWUPGRADE_SCRIPT successfully."
  fi
fi

exit 0
