#!/bin/bash

base_path=$0
function LOG() {
  if [[ ${base_path} =~ '/usr/bin/' ]]; then
    logger "$@"
  else
    command /bin/echo $(date +'%FT%T') "$@"
  fi
}

if [[ ! -f "/proc/device-tree/dcp2" ]]; then
  LOG "Skip dcp404 path upgrade script, not a DCP-2 chassis."
  exit 0
fi

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to persistent directory to update
  -h, --help                       display help
EOM

  exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

PERSISTENT_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift


# Redis-profile
[ -r /etc/profile.d/redis.sh ] && . /etc/profile.d/redis.sh

if [[ -z ${REDIS_CONF_HASH_TAG} ]]; then
  LOG "Undefine: REDIS_CONF_HASH_TAG"
  exit 1
fi

[ -z "${REDIS_PORT}" ] && REDIS_PORT=40001
CLI_BIN="/usr/bin/redis-cli"

LOG "Triggered: ${base_path} REDIS_PORT=${REDIS_PORT}"

REDIS_DATA_PORT=50002

# Fetching mount table
declare -A CONF_MOUNTS
declare -A DATA_MOUNTS
mount_raw=$(redis-cli -c -p ${REDIS_DATA_PORT} dh.get ${REDIS_DATA_HASH_TAG}":so:mounts" | jq --raw-output '.value[] | select(.instance_type=="DATA") | .index,.tag')
mount_raw=( $mount_raw )
for ((i=0; i<${#mount_raw[@]}; i+=2)) { DATA_MOUNTS[${mount_raw[i]}]=${mount_raw[i+1]}; }

mount_raw=$(redis-cli -c -p ${REDIS_DATA_PORT} dh.get ${REDIS_DATA_HASH_TAG}":so:mounts" | jq --raw-output '.value[] | select(.instance_type=="CONF") | .index,.tag')
mount_raw=( $mount_raw )
for ((i=0; i<${#mount_raw[@]}; i+=2)) { CONF_MOUNTS[${mount_raw[i]}]=${mount_raw[i+1]}; }

dcp404_product_name="DCP-404"


for slot_id in {1,2}; do
  # Get slot conf hash and check slot product name
  SLOT_DATA_HASH_TAG=${DATA_MOUNTS[$slot_id]}
  SLOT_CONF_HASH_TAG=${CONF_MOUNTS[$slot_id]}

  slot_product_name=$(${CLI_BIN} -c -p ${REDIS_DATA_PORT} dh.get ${SLOT_DATA_HASH_TAG}:dh/system/info:inventory/productName | jq -r '.value')
  if [[ "$slot_product_name" == "$dcp404_product_name" ]]
  then
    # For R9.0 we updated the way we config application code on transceiver to resemble the DCP-1203 method.
    # For this we need to upkeep some paths since we have split config/application into 2 paths with different functions.
    # (Previously we changed muxponder mode and application code in same command with hardcoding. 
    #  Now they are separate and no longer hardcoded)
    # Therefore, we need to handle upgrade from R7 and R8 to R9+ and downgrade to R7 and R8
    if [[ "$RUNNING_RELEASE_VERSION" =~ "7." ]] || [[ "$RUNNING_RELEASE_VERSION" =~ "8." ]]
    then
      # Get current application and convert to new format
      current_application=$(${CLI_BIN} -c -p ${REDIS_PORT} dh.get ${SLOT_CONF_HASH_TAG}:dh/traffic_manager:interface/5/config/application | jq -r '.value')
      new_service=""
      if [[ "${current_application}" == "400G" ]]; then
        new_service="mux:4x100G-400G"
      elif [[ "${current_application}" == "300G" ]]; then
        new_service="mux:3x100G-300G"
      elif [[ "${current_application}" == "200G" ]]; then
        new_service="mux:2x100G-200G"
      elif [[ "${current_application}" == "200G16QAM" ]]; then
        new_service="mux:2x100G-200G16QAM"
      elif [[ "${current_application}" == "100G" ]]; then
        new_service="mux:1x100G-100G"
      elif [[ "${current_application}" == "100GscFEC" ]]; then
        new_service="mux:1x100G-100GscFEC"
      elif [[ -z "${current_application}" || "${current_application}" == "(nil)" ]]; then
        LOG "No application read from redis. Do nothing."
        exit 0
      else 
        LOG "Unrecognized application value: ${current_application}."
        exit 1
      fi

      new_config="{\"value\":\"${new_service}\"}"

      # Config new service key using old application.
      ${CLI_BIN} -c -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${SLOT_CONF_HASH_TAG} dh/traffic_manager interface/5/config/service ${new_config} &> /dev/null
      ${CLI_BIN} -c -p ${REDIS_PORT} DH.SET               ${SLOT_CONF_HASH_TAG}:dh/traffic_manager:interface/5/config/service ${new_config} &> /dev/null

      # Delete old key
      ${CLI_BIN} -c -p ${REDIS_PORT} DEL ${SLOT_CONF_HASH_TAG}:dh/traffic_manager:interface/5/config/application &> /dev/null
    else
      LOG "No need to handle dcp404 path upgrade from $RUNNING_RELEASE_VERSION for slot ${slot_id}."
    fi

  else 
    LOG "Slot ${slot_id} is not a DCP-404. Skip DCP-404 path upgrades for this slot."
  fi
done

exit 0