#!/bin/bash

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to persistent directory to update
  -h, --help                       display help
EOM

  exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

PERSISTENT_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift

if [ ! -d "$PERSISTENT_DIRECTORY" ]
then
  LOG "The directory $PERSISTENT_DIRECTORY does not exist."
  exit 1
fi

DEFAULT="${PERSISTENT_DIRECTORY}/../image/factory_defaults/etc/network/interfaces"
CURRENT="${PERSISTENT_DIRECTORY}/etc/network/interfaces"
COUNT_BR0_DEFAULT=$(grep "iface br0" ${DEFAULT} | wc -l )
COUNT_BR0_CURRENT=$(grep "iface br0" ${CURRENT} | wc -l )
COUNT_ETH0_DEFAULT=$(grep "iface eth0" ${DEFAULT} | wc -l )
COUNT_ETH0_CURRENT=$(grep "iface eth0" ${CURRENT} | wc -l )

if [[ -f "/proc/device-tree/is_module" ]] ||
   [[ ! -f "/proc/device-tree/is_chassis" ]]; then
  if [[ ${RUNNING_RELEASE_VERSION} =~ ^([0-9]+).[0-9]+.[0-9]+ ]]; then
    MAJOR=${BASH_REMATCH[1]}
    # Preserve network config file which is written by ILA (since R8)
    if (( MAJOR >= 8 )); then
      LOG "Preserve network interfaces file for slot module!"
    elif [[ -s ${CURRENT} ]]; then
      LOG "Empty network interfaces file for slot module!"
      echo -n "" > ${CURRENT}
    fi
  else
    LOG "Invalid version: ${RUNNING_RELEASE_VERSION}"
  fi
elif [[ ${COUNT_BR0_DEFAULT} -ne ${COUNT_BR0_CURRENT} ]] || [[ ${COUNT_ETH0_DEFAULT} -ne ${COUNT_ETH0_CURRENT} ]]; then
  LOG "Incompatible network interface detected!"
  cp -f ${DEFAULT} ${CURRENT}

  if [[ -f /proc/device-tree/dcpr ]]; then
    ifconfig=$(ifconfig eth0.2 | grep "inet addr" | xargs)
    ipaddr=$(echo ${ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
    netmask=$(echo ${ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
    gateway=$(ip route show eth0.2 | grep default | sed 's/.*via\s\(\S*\).*/\1/')
    LOG "Restore defaults network interface, keep eth0.2 network settings: [${ipaddr}] [${netmask}] [${gateway}]"
    # Fresh DCP chassis from factory do not have gateway configured
    if [[ -z ${gateway} ]]; then
      sed -i '/./{H;$!d} ; x ; s/\(iface eth0.2 inet static\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${ipaddr}'\n\tnetmask '${netmask}'/' ${CURRENT}
    else
      sed -i '/./{H;$!d} ; x ; s/\(iface eth0.2 inet static\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${ipaddr}'\n\tnetmask '${netmask}'\n\tgateway '${gateway}'/' ${CURRENT}
    fi
    # delete all leading blank lines at top of file
    sed -i '/./,$!d' ${CURRENT}
  elif [[ -f "/proc/device-tree/is_chassis" ]]; then
    br0_ifconfig=$(ifconfig br0 | grep "inet addr" | xargs)
    br0_ipaddr=$(echo ${br0_ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
    br0_netmask=$(echo ${br0_ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
    br0_gateway=$(ip route show br0 | grep default | sed 's/.*via\s\(\S*\).*/\1/')
    LOG "Restore defaults network interface, keep br0 network settings: [${br0_ipaddr}] [${br0_netmask}] [${br0_gateway}]"

    # Fresh DCP chassis from factory do not have gateway configured
    if [[ -z ${br0_gateway} ]]; then
      sed -i '/./{H;$!d} ; x ; s/\(iface br0 inet static\n.*\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${br0_ipaddr}'\n\tnetmask '${br0_netmask}'/' ${CURRENT}
    else
      sed -i '/./{H;$!d} ; x ; s/\(iface br0 inet static\n.*\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${br0_ipaddr}'\n\tnetmask '${br0_netmask}'\n\tgateway '${br0_gateway}'/' ${CURRENT}
    fi

    # eth0 does not have any gateway configured
    eth0_ifconfig=$(ifconfig eth0 | grep "inet addr" | xargs)
    eth0_ipaddr=$(echo ${eth0_ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
    eth0_netmask=$(echo ${eth0_ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
    LOG "Restore defaults network interface, keep eth0 network settings: [${eth0_ipaddr}] [${eth0_netmask}]"
    sed -i '/./{H;$!d} ; x ; s/\(iface eth0 inet static\n\)\s*address \S\+\s*netmask \S\+/\1\taddress '${eth0_ipaddr}'\n\tnetmask '${eth0_netmask}'/' ${CURRENT}

    # delete all leading blank lines at top of file
    sed -i '/./,$!d' ${CURRENT}
  else
    LOG "Restore defaults network interface."
  fi
fi

