#!/bin/bash

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -r, --runningversion    <release version>        runnning release version
  -i, --installingversion <release version>        installing release version
  -d, --directory         <path>                   path to persistent directory to update
  -h, --help                                       display help
EOM

	exit 2
}

function delete_redis_key () {

  # convert long options to getopt short ones
  for arg in "$@"; do
    shift
    case "$arg" in
      "--port")                set -- "$@" "-p" ;;
      "--key")                 set -- "$@" "-k" ;;
      *)                       set -- "$@" "$arg"
    esac
  done

  cursor="0"
  pattern=""
  port=""
  CLI_BIN="/usr/bin/redis-cli"

  OPTIND=1         # Reset in case getopts has been used previously in the shell.
  while getopts "p:k:" opt; do
    case "$opt" in
      p)  port=$OPTARG
        ;;
      k)  pattern=$OPTARG
        ;;
      esac
  done
  shift $((OPTIND-1))

  while
  mapfile -t matches < <(${CLI_BIN} -c -p ${port} SCAN $cursor MATCH "${pattern}")
  cursor=${matches[0]}
  [[ "${cursor}" -ne "0" ]]
  do
    #Scan shows "(empty array)" on the value paired with cursor.
    #Therefore, we treat the result as and array.
    mapfile -t result < <(echo "${matches[1]}")
    for i in "${!result[@]}"; do
      match=${result[$i]}
      if [[ "${match}" == *"${pattern}" ]]; then
        output="$(${CLI_BIN} -c -p ${port} DEL ${match})"
        echo "Deleted: ${match}"
        if [[ $output -ne 1 ]]; then
          echo "Failed to delete ${match}"
        fi
      fi
    done
  done
}

export -f delete_redis_key

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--runningversion")      set -- "$@" "-r" ;;
    "--installingversion")   set -- "$@" "-i" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

RUNNING_RELEASE_VERSION=''
INSTALLING_RELEASE_VERSION=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?r:i:d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    r)  RUNNING_RELEASE_VERSION=$OPTARG
      ;;
    i)  INSTALLING_RELEASE_VERSION=$OPTARG
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done


[ "${1:-}" = "--" ] && shift

if [ -z $RUNNING_RELEASE_VERSION ]
then
  LOG "Failed to get the running release version"
  exit 1
fi

if [ -z $INSTALLING_RELEASE_VERSION ]
then
  LOG "Failed to get the installing release version"
  exit 1
fi

if [ ! -d "$PERSISTENT_DIRECTORY" ]
then
  LOG "The directory $PERSISTENT_DIRECTORY does not exist."
  exit 1
else
  LOG "Upgrade script will update the persistent content at : $PERSISTENT_DIRECTORY."
fi

if [ ! -f /proc/device-tree/dcpm32csozrplus ]; then
  THROW 1 "This release $INSTALLING_RELEASE_VERSION is only supported on DCPM32CSOZR+"
fi

export RUNNING_RELEASE_VERSION
export INSTALLING_RELEASE_VERSION
SCRIPTS_DIR=$(readlink -f $0 | xargs dirname)
MIGRATE_TO_REDIS=0
if [[ "$RUNNING_RELEASE_VERSION" == "$INSTALLING_RELEASE_VERSION" ]]
then
  LOG "Running version and installing version are the same."
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  REDIS_PORT=30001 START_REDIS=false ${SCRIPTS_DIR}/copy-nw-settings-to-redis.sh
  REDIS_PORT=30001 ${SCRIPTS_DIR}/dcp-upgrades/dcp404-path-upgrade.sh
elif [[ "$RUNNING_RELEASE_VERSION" =~ "6.0." ]]
then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  MIGRATE_TO_REDIS=1
elif [[ "$RUNNING_RELEASE_VERSION" =~ "6.1." ]]
then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  MIGRATE_TO_REDIS=1
elif [[ "$RUNNING_RELEASE_VERSION" == "6.0.2" ]]
then
  LOG "Doing upgrade from 6.0.2"
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
elif [[ "$RUNNING_RELEASE_VERSION" == "6.1.1" ]]
then
  LOG "Doing upgrade from 6.1.1"
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
# Note! Not all upgrade scripts below should be executed when upgrading FROM release 10 and forward. For example "upgrade_dcpr_cloop_wss_ports.sh" MUST NOT be run when upgrading from release 10.
elif [[ "$RUNNING_RELEASE_VERSION" =~ "7." ]] || [[ "$RUNNING_RELEASE_VERSION" =~ "8." ]] || [[ "$RUNNING_RELEASE_VERSION" =~ "9." ]] || [[ "$RUNNING_RELEASE_VERSION" =~ ^9[0-9]{2,}.[0-9]{2,}.[0-9]+ ]]
then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  REDIS_PORT=30001 START_REDIS=false ${SCRIPTS_DIR}/copy-nw-settings-to-redis.sh
  REDIS_PORT=30001 ${SCRIPTS_DIR}/dcp-upgrades/dcp404-path-upgrade.sh
  ${SCRIPTS_DIR}/upgrade_lldp.sh
  ${SCRIPTS_DIR}/upgrade_dcpr_cloop_wss_ports.sh
  ${SCRIPTS_DIR}/upgrade_dcp2_automation_mode.sh
else
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  LOG "Do not know upgrade path"
fi

 REDIS_PORT=30001 ${SCRIPTS_DIR}/remove-password-in-redis.sh
 ${SCRIPTS_DIR}/update-tacacs-radius.sh -d $PERSISTENT_DIRECTORY
 ${SCRIPTS_DIR}/update-syslog-ng-config.sh -d $PERSISTENT_DIRECTORY

# Check if smartmanager is available or not
[[ -f /usr/sbin/smartmanager/smartmanager.js ]] && MIGRATE_TO_REDIS=1

if [[ ${MIGRATE_TO_REDIS} -ne 0 ]]; then
  ${SCRIPTS_DIR}/migrate/migrate-to-redis.sh -d $PERSISTENT_DIRECTORY
  ret=$?
  if [ $ret -ne 0 ]; then
    LOG "Script failed: migrate-to-redis.sh, code: $ret"
    exit 1
  fi
else
   REDIS_PORT=30001 ${SCRIPTS_DIR}/handle-shadow-user-accounts.sh
fi

exit 0
