#!/bin/bash

base_path=$0
function LOG() {
  if [[ ${base_path} =~ '/usr/bin/' ]]; then
    logger "$@"
  else
    command /bin/echo $(date +'%FT%T') "$@"
  fi
}

if [[ ! -f "/proc/device-tree/is_chassis" ]]; then
  LOG "Skip network settings for this module!"
  exit 0
fi

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to persistent directory to update
  -h, --help                       display help
EOM

  exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

PERSISTENT_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift


# Redis-profile
[ -r /etc/profile.d/redis.sh ] && . /etc/profile.d/redis.sh

if [[ -z ${REDIS_CONF_HASH_TAG} ]]; then
  LOG "Undefine: REDIS_CONF_HASH_TAG"
  exit 1
fi

[ -z "${START_REDIS}" ] && START_REDIS=true
[ -z "${REDIS_PORT}" ] && REDIS_PORT=40001
CLI_BIN="/usr/bin/redis-cli"

LOG "Triggered: ${base_path} START_REDIS=${START_REDIS} REDIS_PORT=${REDIS_PORT}"

function ping_redis() {
  timer=30
  redis_port=${1}
  is_up=false
  while true; do
      # Check if redis is up
      status=$(redis-cli -p ${redis_port} PING)
      if [[ "${status}" == 'PONG' ]]; then
          is_up=true
          break
      fi
      timer=$((timer-1))
      LOG "Waiting for redis... timeout="${timer}
      if [[ ${timer} == 0 ]]; then
        is_up=false
        break
      fi
      sleep 1
  done
  echo ${is_up}
}

function wait_redis_persistent() {
  timer=180
  while true; do
      # Check if the write operation is finished
      status=$(${CLI_BIN} -p ${REDIS_PORT} info persistence | grep -i aof_rewrite_in_progress)
      if [[ ${status} =~ ':0' ]]; then
          LOG "Redis BGREWRITEAOF finished!"
          break
      fi
      timer=$((timer-1))
      LOG "Waiting for BGREWRITEAOF... timeout="${timer}
      if [[ ${timer} == 0 ]]; then
          LOG "Failed to save AOF file!"
          exit 1
      fi
      sleep 1
  done
}

if [[ ${START_REDIS} == true ]]; then
  WORKSPACE="${PERSISTENT_DIRECTORY}/etc/sw_storage/redis/"
  if [[ ! -d "$PERSISTENT_DIRECTORY" ]]; then
    LOG "The directory $PERSISTENT_DIRECTORY does not exist."
    exit 1
  fi
  mkdir -p ${WORKSPACE}
  SW_STORAGE=`readlink -f ${WORKSPACE}`
  LOG "Workspace: ${WORKSPACE} -> ${SW_STORAGE}"

  # Start redis-server
  redis-server /etc/redis/50001.conf --port ${REDIS_PORT} --cluster-enabled no --dir ${SW_STORAGE} &>/dev/null&
  sleep 1s

  PING_REDIS=`ping_redis ${REDIS_PORT}`
  REDIS_VERSION=`redis-server --version`
  if [[ ${PING_REDIS} == true ]]; then
    LOG "Redis is up and running at port: ${REDIS_PORT}, version: ${REDIS_VERSION}"
    sleep 1s # Waiting for some unexpected paths to be loaded
  else
    LOG "Failed to start redis server at port: ${REDIS_PORT}, version: ${REDIS_VERSION}"
    exit 1
  fi
fi

mgmt_ip=""
mgmt_netmask=""
mgmt_gateway=""
if [[ -f /proc/device-tree/dcpr ]] || [[ -f /proc/device-tree/dcpm2 ]]; then
  ifconfig=$(ifconfig eth0.2 | grep "inet addr" | xargs)
  mgmt_ip=$(echo ${ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
  mgmt_netmask=$(echo ${ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
  mgmt_gateway=$(ip route show eth0.2 | grep default | sed 's/.*via\s\(\S*\).*/\1/')
  LOG "Collected eth0.2 network settings: [${mgmt_ip}] [${mgmt_netmask}] [${mgmt_gateway}]"
elif [[ -f "/proc/device-tree/is_chassis" ]]; then
  br0_ifconfig=$(ifconfig br0 | grep "inet addr" | xargs)
  mgmt_ip=$(echo ${br0_ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
  mgmt_netmask=$(echo ${br0_ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
  mgmt_gateway=$(ip route show br0 | grep default | sed 's/.*via\s\(\S*\).*/\1/')
  LOG "Collected br0 network settings: [${mgmt_ip}] [${mgmt_netmask}] [${mgmt_gateway}]"
fi

# Both DCP-R and DCP-2/DCP-M are using same redis conf path for mgmt-ip config: config/br0/ipv4Settings, is that a bug?
mgmt_config="{\"value\":{\"ipAddress\":\"${mgmt_ip}\",\"netmask\":\"${mgmt_netmask}\",\"gateway\":\"${mgmt_gateway}\"}}"
LOG "Creating config Prefix: dh/network Path: config/br0/ipv4Settings Value: ${mgmt_config}"
${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/network config/br0/ipv4Settings ${mgmt_config}
${CLI_BIN} -p ${REDIS_PORT} DH.SET               ${REDIS_CONF_HASH_TAG}:dh/network:config/br0/ipv4Settings ${mgmt_config}

# Both DCP-R and DCP-2/DCP-M use same local eth0 nw interface
# eth0 does not have any gateway configured
eth0_ifconfig=$(ifconfig eth0 | grep "inet addr" | xargs)
eth0_ipaddr=$(echo ${eth0_ifconfig} | sed 's/.*addr:\(\S*\).*/\1/')
eth0_netmask=$(echo ${eth0_ifconfig} | sed 's/.*Mask:\(\S*\).*/\1/')
LOG "Collected eth0 network settings: [${eth0_ipaddr}] [${eth0_netmask}] "
eth0_config="{\"value\":{\"ipAddress\":\"${eth0_ipaddr}\",\"netmask\":\"${eth0_netmask}\",\"gateway\":\"\"}}"
LOG "Creating config Prefix: dh/network Path: config/eth0/ipv4Settings Value: ${eth0_config}"
${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/network config/eth0/ipv4Settings ${eth0_config}
${CLI_BIN} -p ${REDIS_PORT} DH.SET               ${REDIS_CONF_HASH_TAG}:dh/network:config/eth0/ipv4Settings ${eth0_config}

# Save DNS settings
cat /etc/resolv.conf | {
  read -r line
  dns1="${line#nameserver }"
  read -r line
  dns2="${line#nameserver }"
  LOG "Collected DNS settings: [${dns1}] [${dns2}]"
  dns_config="{\"value\":{\"primaryDnsIpAddress\":\"${dns1}\",\"secondaryDnsIpAddress\":\"${dns2}\"}}"
  LOG "Creating config Prefix: dh/network Path: config/dnsSettings Value: ${dns_config}"
  ${CLI_BIN} -p ${REDIS_PORT} DH.CREATE_CONFIG_KEY ${REDIS_CONF_HASH_TAG} dh/network config/dnsSettings "${dns_config}"
  ${CLI_BIN} -p ${REDIS_PORT} DH.SET               ${REDIS_CONF_HASH_TAG}:dh/network:config/dnsSettings "${dns_config}"
}

if [[ ${START_REDIS} == true ]]; then
  redis-cli -p ${REDIS_PORT} BGREWRITEAOF
  wait_redis_persistent
  redis-cli -p ${REDIS_PORT} CONFIG SET auto-aof-rewrite-percentage 0
  wait_redis_persistent

  # Shutdown redis-server
  redis-cli -p ${REDIS_PORT} SHUTDOWN SAVE
fi

exit 0
